#!/usr/bin/env python
import pika
import time
import websocket, base64
import simplejson as json
import os
import sys
import logging_rpi
import requests
import subprocess
from datetime import datetime
import BackhaulConfiguration

# SocketWorkerCB is RabbitMQ(pika)'s call back function and the last point for data before it goes to Pulsar
# For any data, it will go through customCallback and retry data sending at most 3 times before sleeping
# Data must be acknowledged with ack or reject -- otherwise the data is never discarded

# If backhaul enables direct reply, customCallback will requeue the server's response back to an imaginary queue

# Credits to Tymm lmao

class SocketWorkerCB(object):

	def __init__(self):

		# Backhaul Configuration
		BackhaulConfiguration.generateConfigurationFile()
		
		self.configuration = BackhaulConfiguration.getDeviceConfig()
		self.updateInterval = self.configuration["UpdateInterval"]
		self.restartInterval = self.configuration["RestartInterval"]
		self.channel = None
		self.connection = None

		# SocketCB Variables
		self.topicPath = '/ws/v2/producer/persistent/public/default/'
		self.publicQueue = 'publicQueue'
		self.wsDict = {}
		self.firstTime = True
		self.socketErrors = ['10053', 'Broken pipe', 'Connection is already closed', 'Connection timed out']
		self.lastUpdated = datetime.now()
		self.lastHeartbeat = datetime.now()

		# test rabbit serve
		self.generalObj = {}
		self.generalObj["Topic"] = "generic-test-topic"
		self.generalObj["Data"] = {}
		self.generalObj["Data"]["DummyData"] = 123
		self.headers = {'content-type': 'application/json'}

		# Websocket Type
		if self.configuration["SendType"] == "Websocket":
			self.connection = pika.BlockingConnection(pika.ConnectionParameters(host='localhost'))
			self.channel = self.connection.channel()
			self.channel.queue_declare(queue="websocketQueue", durable=True)
			self.channel.confirm_delivery()
			logging_rpi.log.info("socketWorkerCB Module:::connect to websocketQueue:::Successful")
		else:
			self.wsReaderURL = None
			self.wsReaderConn = None

		# Pulsar Topic
		self.baseTopic = None
		self.updateBaseTopic()
		print("Connected to: " + str(self.baseTopic))

	def updateBaseTopic(self):

		# First: update Pulsar address in the conf file
		self.configuration = BackhaulConfiguration.getDeviceConfig()

		try:
			updateStatus = BackhaulConfiguration.updatePulsarAddress()
			if not updateStatus:
				logging_rpi.log.error("SocketWorkerCB Module:::Server Address not Updated:::Possible Timeout. SocketWorkerCB will use previous address if available.")
		except Exception as e:
			logging_rpi.log.error("SocketWorkerCB Module:::Server Address not Updated:::" + str(e))
			logging_rpi.log.warn("SocketWorkerCB Module:::Server Address not Updated:::Failed. Using old Pulsar Address if exist")

		# Second: get the address from the config file
		# If it is not updated, then try getting a previous one if got

		address = BackhaulConfiguration.getPulsarAddress()
		if address is not None:
			self.baseTopic = address + self.topicPath
		else:
			# see if baseTopic got previous data or not
			if self.baseTopic is not None:
				logging_rpi.log.warn("SocketWorkerCB Module:::Server Address not Updated:::Failed. Using old Pulsar Address if exist")
			else:
				logging_rpi.log.info("SocketWorkerCB Module:::Server Address not Updated:::No valid pulsar address. Sleeping for seconds: " + str(self.restartInterval))
				time.sleep(self.restartInterval)
				return self.updateBaseTopic()

	def createNewSocket(self, topic):
		# Create websocket-client connection to send data to Pulsar
		# Pulsar Server Address (ws://....:80 + nameplace, etc) + Topic Name
		if topic not in self.wsDict:
			ws = websocket.create_connection(self.baseTopic + topic)
			self.wsDict[topic] = ws
			print("Topic : " + topic + " has been created")

	def reconnectSocket(self, topic):
		# Connections can drop from inactivity, leading to broken pipe or so. 
		# Reconnect by creating the connection again
		if topic in self.wsDict:
			self.wsDict[topic].close()
			logging_rpi.log.info("Topic : " + topic + " closed")
		else:
			logging_rpi.log.info("Topic : " + topic + " has already been closed")

		self.wsDict[topic] = websocket.create_connection(self.baseTopic + topic)
		logging_rpi.log.info("Socket for " + topic + " has been reconnected")

	def customCallback(self, ch, method, properties, body, retries):

		if retries < 0:
			return False

		try:
			#Handle old data format and new 
			data = json.loads(body)
			directReply = False
			try:
				testOne = data["Topic"]
				testTwo = data["Data"]
			except KeyError as e:
				data["Topic"] = data.pop("topic")
				data["Data"] = data.pop("data")
				logging_rpi.log.warn("SocketWorkerCB Module:::Old Format potentially is used:::Handled.")

			# Backhaul will only update itself according to the interval set, starting from the beginning at this script
			# If more than X seconds pass, update itself by calling an API from the Server
			# Also saves a backup in case the actual one dies from corruption

			currentTime = datetime.now()
			difference =  (currentTime - self.lastUpdated).seconds
			heartbeatDifference =  (currentTime - self.lastHeartbeat).seconds
			if difference >= self.updateInterval:
				print("Interval hit. Updating configuration...")
				self.updateBaseTopic()
				self.lastUpdated = currentTime
				self.updateInterval = self.configuration['UpdateInterval']
				self.restartInterval = self.configuration['RestartInterval']
				self.reconnectSocket(data['Topic'])
				BackhaulConfiguration.writeBackup()
				BackhaulConfiguration.logProcess(os.getpid()) # will only log PID if windows

				print(" + + + + + + + + + + + + + + +")
				print("Update Interval is now: " + str(self.updateInterval))
				print("Restart Interval is now: " + str(self.restartInterval))
				logging_rpi.log.info("SocketWorkerCB Module:::Pulsar Address Updated:::Changed to: "+str(self.baseTopic))
				print(" + + + + + + + + + + + + + + +")
			
			if (heartbeatDifference >= self.configuration['HeartbeatInterval'] or self.firstTime) and self.configuration["Device"] == "Server":
				print("Sending Heartbeat!")
				self.lastHeartbeat = currentTime
				# emit heartbeat here
				x = requests.post("http://127.0.0.1:{0}/queueRabbitMQ".format(self.configuration["RMQServerPort"]), json = self.generalObj, headers=self.headers)
				if x.text == "200":
					package = {}
					package["ServiceId"] = self.configuration["ServiceMonitorID"]
					url = "{0}monitor/ping?payload={1}".format("http://service-monitor.metatechnology.co.uk/", json.dumps(package))
					req = requests.get(url)
					if req.status_code == 200:
						logging_rpi.log.info("SocketWorkerCB Module:::Pinged Service Monitor:::")
					else:
						logging_rpi.log.error("SocketWorkerCB Module:::Unable to Ping Service Monitor:::")
				else:
					print("RabbitMQ Failed")
				# logging here 
				if os.path.exists("backhaulHeartbeat"):
					createTime = os.path.getctime("backhaulHeartbeat")
					current_time = time.time()
					if (current_time - createTime) // (24 * 3600) > 1:
						open('backhaulHeartbeat', 'w').close()
					print("file age: " + str(createTime))
				with open('backhaulHeartbeat', 'w+') as f:
					f.write(str(datetime.now()))
					f.write(" || PROCESS ID: " + str(os.getpid()))
				BackhaulConfiguration.cleanLogDirectory()

			# Create socket with the Pulsar URL + Topic Name
			
			try:
				directReply = data.pop('DirectReply')
			except Exception as e:
				print(e)
				directReply = False
				pass

			# If websocket is enabled and topics are enabled
			if self.configuration['SendType'] in ["Websocket", "Both"] and data['Topic'] in self.configuration['WebsocketTopics']:
				self.channel.basic_publish(
					exchange='',
					routing_key="websocketQueue",
					properties=pika.BasicProperties(
						delivery_mode=2  # make message persistent
					),
					body=json.dumps(data),
					mandatory=True
				)	

				if directReply and self.configuration['SendType'] == "Websocket":
					ch.basic_publish('', routing_key=properties.reply_to, body="ok")
					print('Sent to Websocket Queue.')

			print("- - - - - - - - - - - - - -")

			#If "Both" or Pulsar, send. Otherwise, if it's websocket's other functions
			if self.configuration['SendType'] != "Websocket" or (self.configuration['SendType'] == "Websocket" and data['Topic'] not in self.configuration['WebsocketTopics']):
				# if its both or pulsar
				# websocket, but topic is not meant for websocket
				self.createNewSocket(data['Topic'])
				jsd = json.dumps({
					'payload' : base64.b64encode(json.dumps(data['Data']).encode('utf-8')).decode('utf-8'),
					'context' : 5
				})
				self.wsDict[data['Topic']].send(jsd)
				res = self.wsDict[data['Topic']].recv()
				if self.wsDict[data['Topic']].connected:
					response = json.loads(res)
					if response['result'] == 'ok':
						ch.basic_ack(delivery_tag=method.delivery_tag)
						print("PULSAR: {0} published".format(data["Topic"]))
						print(" - - - - - - - - - - - - - - - - - - ")
						if self.firstTime:
							BackhaulConfiguration.writeBackup()
							self.firstTime = False
						if directReply and self.configuration['SendType'] in ["Pulsar", "Both"]:
							ch.basic_publish('', routing_key=properties.reply_to, body=res)
							print('DirectReply published successfully')
						return True
					else:
						print('Failed to publish message:', response)
				else:
					self.reconnectSocket(data['Topic'])
			else:
				# Not our things, don't send
				print "[X} " + data['Topic'] + " to WEBSOCKET"
				ch.basic_ack(delivery_tag=method.delivery_tag)
				return True

			print("Not successfully sent...")
			retries -= 1
			return self.customCallback(ch, method, properties, body, retries)
		except Exception as e:
			logging_rpi.log.error("SocketWorkerCB:::Exception Occured:::"+str(e))
			print("Exception as e : " + str(e))
			if any(err in str(e) for err in self.socketErrors):
				print("Socket-related errors : " + str(e))
			retries -= 1
			print("Retries left: " + str(retries))
			self.reconnectSocket(data['Topic'])
			return self.customCallback(ch, method, properties, body, retries)

	def main(self):
		#Set up configuration before consuming
		#Function is what it will run when it gets a message
		self.connection = pika.BlockingConnection(pika.ConnectionParameters(host='localhost'))
		self.channel = self.connection.channel()
		self.channel.queue_declare(queue=self.publicQueue, durable=True)

		print(' [*] Waiting for messages. To exit press CTRL+C')

		def callback(ch, method, properties, body):
			if not self.customCallback(ch, method, properties, body, 3):
				#Sleeps the script if the data is unable to be sent
				logging_rpi.log.error("SocketWorkerCB:::Publishing Failed:::Exhausted Retry Attempts. Sleeping socketCB for 60 seconds.")
				time.sleep(60)
				ch.basic_reject(delivery_tag=method.delivery_tag, requeue=True)
		
		self.channel.basic_qos(prefetch_count=1)
		self.channel.basic_consume(queue=self.publicQueue, on_message_callback=callback)

		# After set up, actually start running the consuming

		print("----------------------------------- ccb -----------------------------------")
		self.channel.start_consuming()

if __name__== "__main__":

	restartInterval = BackhaulConfiguration.getRestartInterval()
	BackhaulConfiguration.logProcess(os.getpid())

	# Runs forever
	while True:
		try:
			swcb = SocketWorkerCB()
			swcb.main()
		except KeyboardInterrupt as e:
			exit()
		except Exception as e:
			print(e)
			if str(e) == "cannot concatenate 'str' and 'int' objects":
				logging_rpi.log.error("SocketWorkerCB:::Fatal Error::: Invalid Pulsar Address | Sleeping for {0} seconds before resuming operation.".format(restartInterval))
			else:
				logging_rpi.log.error("SocketWorkerCB:::Fatal Error::: Unexpected Error | {0} | Sleeping for {1} seconds before resuming operation.".format(str(e), restartInterval))
			time.sleep(restartInterval)
			pass


